#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define F_CPU 20000000
#include <util/delay.h>

#include "RS232LIB.h"
#include "FORMATLIB.h"
#include "CORDIC.h"
#include "DCFLIB.h"

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

// Use compiler option -O1

// port b0 pin14 = clipled

// D.3  pin 5 is  OC2B
// used here for 1 MHz output


// show ISR timing on PORTB.2 pin 16
// PORTB.7 pin 10


#define IO2pin 3
#define IO2port PORTD

#define IO3pin 2
#define IO3port PORTB

#define IO4pin 7
#define IO4port PORTB



// Frontend parameters
//
// Md   number of input-sequence samples used until one I/Q-integration output 
//      value is generated by the ISR and put into the FIFO
//      values go at half-rate into downsampling two-stage CIC filters
//
// Ns   number of clock cycles between ADC sampling
//
// Mc   order of the smoothing CIC filter behind the downsampling
//
// P    the ouput of doIQsample routine from the FIFO is divided by P before
//      it enters the second CIC-Filter. This is used to prevent overflow
//      in the second CIC filter and in the CORDIC subroutine.
//
//      for scaling of frontend for correct uMean computation:
//      use :     (Mc/P) * (Md^2/2^2) == 32
//      for fast division P should be a power of 2
//       P=Mc/32 * (Md^2/2^2) = =Mc * Md^2  / 128


// Frontend parameters
  #define Md  16
  #define Ns 2500UL
  #define Mc 8 
  #define OCRoffset 1250
  #define P 16
  #define FRQlength 8
  #define FRQdivider 1
  #define HEADLINE "TDF 162"


//----------------------------------------------------------------------------------------

uint8_t Clip ;
uint8_t ClipCount ;

#define ClipLed 0

void doClipCheck(){
  // CLIP-FLAG is set by DC routine doIQsample() if input out of range
  if (Clip) { PORTB |= _BV(ClipLed) ; } else { PORTB &= ~ _BV(ClipLed) ;}
  ClipCount++ ;
  // reset CLIP-FLAG all 256 steps
  if (ClipCount==0){ Clip=0 ; }
  }

//----------------------------------------------------------------------------------------

// signal stuff

uint16_t sampleTime ;
int16_t ADCv ;
volatile int8_t DAC1,DAC2 ;

int16_t Amplitude,LogAmplitude ;
uint8_t CordicPhase ;
uint8_t LastCordicPhase ;
int8_t  PhaseDifference ;
int16_t ExtendedPhase ;


//----------------------------------------------------------------------------------------

typedef int32_t CICtype  ;

volatile CICtype IIintegrator1 ;
volatile CICtype QQintegrator1 ;

volatile CICtype IIintegrator2 ;
volatile CICtype QQintegrator2 ;

volatile CICtype  IIsample ;
volatile CICtype  QQsample ;

#define IntFifoSize 8
#define IntFifoMask 0x07

volatile uint8_t IntFifoInPtr ;
volatile uint8_t IntFifoOutPtr ;
volatile CICtype  IntFifoII[IntFifoSize] ;
volatile CICtype  IntFifoQQ[IntFifoSize] ;

//----------------------------------------------------------------------------------------

// for 500Hz sample generation
volatile uint8_t ClkMail500 ;
#define TTT 40000
uint16_t Ticks500 ;

void timer1Init(){

  TCCR1A=0b10000010 ; // fast pwm
  TCCR1B=0b00011001	; // ck=XTAL
  // Interupt enable for Timer1-Overflow
  TIMSK1=0b00000001 ;
  OCR1A=OCRoffset ;
  ICR1=Ns-1 ; // sample period predivider

  IntFifoInPtr=0 ;
  IntFifoOutPtr=0 ;
  
  IIintegrator1=0 ;
  QQintegrator1=0 ;
  IIintegrator2=0 ;
 
  QQintegrator2=0 ;

  sampleTime=0 ;
  CordicPhase=0 ;
  LastCordicPhase=0 ;
  ExtendedPhase=0 ;



  }


ISR(TIMER1_OVF_vect) {
   IO3port |=  _BV(IO3pin) ;
// get sample from input and convert to signed value
  ADCv=( (int16_t)ADC-512) ; 
// clip ADC value in order to detect ADC overflow
#define ADCMAX 500 
  if (ADCv > ADCMAX) { ADCv=ADCMAX ; Clip=1 ; }
  if (ADCv <-ADCMAX) { ADCv=-ADCMAX ;  Clip=1 ; }
// ADCv now is in -ADCMAX..ADCMAX

// multiply and sum samples with sequence 1  0  -1  0  1  0  -1  0  1  0 ... for I channel
// multiply and sum samples with sequence 0  1  0  -1  0  1  0  -1  0  1 ... for Q channel
// the summing directly is done as two stage intergration of a CIC filter.
// The CIC integration outputs are in IcicInt2 and QcicInt2

  if (sampleTime & 1) {
    if (sampleTime & 0b10) { IIintegrator1 += ADCv ; } else { IIintegrator1 -= ADCv ; }
	IIintegrator2 += IIintegrator1 ;
    }
   else {
    if (sampleTime & 0b10) { QQintegrator1 += ADCv ; } else { QQintegrator1 -= ADCv ; }
	QQintegrator2 += QQintegrator1 ; 
	}

  sampleTime++ ;
  uartServe() ;
  OCR2A = DAC1 ;
  OCR2B = DAC2 ;

// after Md samples output I/Q values to FIFO 

  if (sampleTime==Md) {
    sampleTime=0 ;
// put CIC integration values downsampled by Md into FIFOs
    IntFifoII[IntFifoInPtr]=IIintegrator2 ;
	IntFifoQQ[IntFifoInPtr]=QQintegrator2 ;
	IntFifoInPtr=(IntFifoInPtr+1) & IntFifoMask ;
    }
   PORTD &= ~ _BV(2) ;  
  Ticks500 += Ns ;
  if(Ticks500>=TTT) {
 	ClkMail500=1 ;  
//	 ticks500++ ;
	Ticks500 -= TTT ;
	PORTD |= _BV(2) ;
    } 



  IO3port &= ~ _BV(IO3pin) ;
  }

//----------------------------------------------------------------------------------------

CICtype IIcomb1store,IIcomb1out ;
CICtype IIcomb2store,IIcomb2out ;
CICtype QQcomb1store,QQcomb1out ;
CICtype QQcomb2store,QQcomb2out ;



int32_t IIfifo[Mc] ;
int32_t IIintegrator3 ;
int32_t IIcic3out ;

int32_t QQfifo[Mc] ;
int32_t QQintegrator3 ;
int32_t QQcic3out ;

uint8_t CICfifoPTR ;

void CICfifoINIT(){
  for ( CICfifoPTR=0 ; CICfifoPTR<Mc ; CICfifoPTR++){
    QQfifo[CICfifoPTR]=0 ;
    IIfifo[CICfifoPTR]=0 ;
    }
  CICfifoPTR=0 ;
  IIintegrator3=0 ;
  QQintegrator3=0 ;
  }


int32_t FRQfifo[FRQlength] ;
int32_t FRQintegrator ;
int32_t FRQcicout ;
uint8_t FRQfifoPTR ;


int8_t PhaseDifference ;
int8_t FrequencyEstimate ;

int8_t BitLine ;

void FRQdemodInit(){
  for ( FRQfifoPTR=0 ; FRQfifoPTR<FRQlength ; FRQfifoPTR++){
    FRQfifo[FRQfifoPTR]=0 ;
    }
  FRQfifoPTR=0 ;
  FRQintegrator=0 ;
  }


// doIQsample() fetches data from the I and Q FIFO
// these data are feed to the 2 stage COMB section of the CIC downsampling filter
// Phase and amplitude are then computed via a CORDIC-like rotation method

void doIQsample(){

// fetch data from FIFO
  IIsample=IntFifoII[IntFifoOutPtr] ;
  QQsample=IntFifoQQ[IntFifoOutPtr] ;
  IntFifoOutPtr=(IntFifoOutPtr+1) & IntFifoMask ;

// do 1.st comb-step of two stage downsampling CIC filter
  IIcomb1out=IIcomb1store-IIsample ; IIcomb1store=IIsample ; 
  QQcomb1out=QQcomb1store-QQsample ; QQcomb1store=QQsample ; 

// do 2.nd comb-step of two stage downampling CIC filter
  IIcomb2out=IIcomb2store-IIcomb1out ; IIcomb2store=IIcomb1out ; 
  QQcomb2out=QQcomb2store-QQcomb1out ; QQcomb2store=QQcomb1out ; 

// IcombOutb and QcombOutb are now the downsampled filtered I and Q channel values.
// these go now to a further 1-stage smoothing CIC filter of order Mc,
// but now further downsampling occurs

// integration step of smoothing CIC filter I-channel
   IIintegrator3 += IIcomb2out / P ;
// comb staep of smoothing CIC filter
   IIcic3out = IIintegrator3 - IIfifo[CICfifoPTR] ;
// FIFO update of smoothing CIC filter
  IIfifo[CICfifoPTR] =  IIintegrator3 ;

// same for I-channel
  QQintegrator3 += QQcomb2out / P ;
  QQcic3out = QQintegrator3 - QQfifo[CICfifoPTR] ;
  QQfifo[CICfifoPTR] =  QQintegrator3 ;
// advance pointer and bump around
  CICfifoPTR++ ; if ( CICfifoPTR==Mc ) { CICfifoPTR=0 ; }

// CICfifoIout and CICfifoQout are now the triple filtered I and Q channel values
// these are used to compute phase and amplitude using a CORDIC maethod



  CordicPhase=  iCordic( IIcic3out , QQcic3out ) ;
  Amplitude=GetCordicAmplitude() ;


  LogAmplitude=Log2(Amplitude+1) ; 



  PhaseDifference=CordicPhase-LastCordicPhase ;
  LastCordicPhase=CordicPhase ;
  ExtendedPhase += PhaseDifference ;

  FrequencyEstimate = PhaseDifference ;

// filter the frquency-signal with a CIC filter
  FRQintegrator += FrequencyEstimate ;
  FRQcicout = FRQintegrator - FRQfifo[FRQfifoPTR] ;
  FRQfifo[FRQfifoPTR] =  FRQintegrator ;
// advance pointer and bump around
  FRQfifoPTR++ ; if ( FRQfifoPTR==FRQlength ) { FRQfifoPTR=0 ; }
// normalize
  
 //  FRQcicout= FRQcicout / FRQdivider ;
 

  //if (uartStart) { DAC1=60 ; uartStart=0 ; } else { DAC1=uartTimer*4 ; }
 // DAC2=128  + FRQcicout  ;
 
/*
  int16_t x1=(LogAmplitude-390)/4 ;
  if (x1<0) { x1=0 ; }
  if (x1>255) { x1=255 ; }
  DAC1= x1 ;
  DAC2= 128+ FrequencyEstimate ;
//  DAC2= 128+CordicPhase ;

*/
  }


//----------------------------------------------------------------------------------------

/*
volatile uint8_t ClkMail ;
uint8_t Clk ;

void timer2Init(){
  // Timer 2 BitClk
  TCCR2A=0b00000010 ; // ctc mode 2
  //  TCCR2B=0b00000111 ; // prescale by 1024
  //  TCCR2B=0b00000110 ; // prescale by 256
  TCCR2B=0b00000100 ; // prescale by 64
  // Parameter for 20000000Hz XTAL
  OCR2A=125-1 ; //  top
  OCR2B=100 ; //  compare
// Interupt enable for Timer2-compare-B
  TIMSK2=0b00000100 ;
  Clk=0 ;
  ClkMail=0 ;
  }

ISR(TIMER2_COMPB_vect) {  
  Clk++ ;
  if (Clk==5) { Clk=0 ; ClkMail=1 ; } 
  }
*/

//volatile uint8_t ClkMail500 ;
//#define TTT 40000
//uint16_t Ticks500 ;


void timer2Init(){
  // Timer 2 BitClk
  TCCR2A=0b10100011 ; // fast pwm top=0xFF OC2A=pin 17/28 OC2B=pin 5/28
 // TCCR2B=0b00000010 ; // prescale by 8 
   TCCR2B=0b00000001 ; // prescale by 1 
  OCR2A=50; 
  OCR2B=150 ; 
// Interupt enable for Timer2-overflow
 // TIMSK2=0b00000001 ; // TOIE2
 // ClkMail500=0 ;
 
  }



/*
ISR(TIMER2_OVF_vect) {
  PORTD &= ~ _BV(2) ;  
  Ticks500 += 2048 ;
  if(Ticks500>=TTT) {
 	ClkMail500=1 ;  
//	 ticks500++ ;
	Ticks500 -= TTT ;
	PORTD |= _BV(2) ;
    } 
  }
*/



//----------------------------------------------------------------------------------------  

int16_t SilenceTimer ;
int16_t SecondTimer ;
int16_t x0,x1,x2 ;

// invoked with 500 samples/sec 

#define TDFlength 50

int16_t TDFbuffer[TDFlength] ;
int16_t TDFintegrator ;

uint8_t TDFptr ;

uint8_t TheSecond ;
uint8_t TheBit ;




uint8_t ptr ;

inline void advance(uint8_t d){  
  ptr += d ;
  if ( ptr >= TDFlength ) { ptr -= TDFlength ; }
  }
int index1='0' ;

int16_t MatchedFilter(int16_t Frequency){

if(0){
  uartPutc('}') ;
  uartPutc(index1) ;
  uartWord(Frequency) ;
  }


  index1++ ;
  if(index1>'9') index1='0' ;

  int16_t sum ;
  // CIC implementation of matched filter
  ptr=TDFptr ;
  // ptr points to oldest sample in TDFbuffer
  sum=-TDFbuffer[ptr] ;
  advance(12) ;
  sum += 2*TDFbuffer[ptr] ;
  advance(25) ;
  sum -= 2*TDFbuffer[ptr] ;
  advance(12) ;
  sum +=   TDFbuffer[ptr] ;

  TDFintegrator += Frequency ;
  // replace oldest sample in buffer
  TDFbuffer[TDFptr]=TDFintegrator ;
  TDFptr++ ;
  if ( TDFptr==TDFlength) {
    TDFptr=0 ;
	}
	
  DAC2=128+sum/16 ;

  return sum ;

  }


void TDFinit(){
  TDFptr=0 ;
  TheSecond=0 ;
  TDFintegrator=0 ;
  }

//#define THRESHOLD 600

#define THRESHOLD 300


void searchForPeak(){
  if ( ( x1>=x0 ) && (x1>x2) && (x1>THRESHOLD) ) { 
    SecondTimer=0 ; 
    SilenceTimer=0 ;
    TheSecond=0 ;
    DisplayTime() ;
	}
  }

void checkForGap(){
  SilenceTimer++ ;
  if ( ( x1>THRESHOLD ) & ( SilenceTimer<390) )  { 
    SilenceTimer=0 ; }
  if ( SilenceTimer>400 ) {
    SilenceTimer=400 ;
    searchForPeak() ;
    }
 //   DAC1=SilenceTimer/4 ;
  }

void checkSecondTimer(){
  SecondTimer++ ;
  if (SecondTimer>=500) {
    SecondTimer=0 ; 
	}

  DAC1= SecondTimer/4  ;
  if ( SecondTimer>50 ) { DAC1= SecondTimer/4+40  ; } 

  if (SecondTimer==50){
    if ( x1>500 ) { 
	  TheBit=1 ; 
	  //IO2port |=  _BV(IO2pin) ; 
	  }
	 else {
	  TheBit=0 ;
	  //IO2port &= ~ _BV(IO2pin) ;
	  }
    uartPutc('0'+TheBit) ;
    putDCFbit(TheSecond,TheBit) ; 
   	TheSecond++ ;
    }
  }

void TDFdemod(){
  int16_t Frequency ;
// get frequency in atomic way
  cli() ;
  Frequency=FRQcicout ;
  sei() ;
  x2=x1 ; x1=x0 ; x0=MatchedFilter(Frequency) ;
  checkForGap() ;
  checkSecondTimer() ;
  }

//----------------------------------------------------------------------------------------
/*
void timer0Init(){
#define T0divider 20  
  TCCR0A=0b10100011 ; // mode 3 , fast pwm TOP=255 out OC0A and OC0B
  TCCR0B=0b00000001	; // ck=XTAL , 78.125kHz
  OCR0A = 50 ;
  OCR0B = 100 ;
  }
*/
void timer0Init(){
// 162 kHz
  TCCR0A=0b00100011 ; // fast pwm  output OCR0B OC0B=PD5 pin 11of28
  TCCR0B=0b00001001	; // ck=XTAL
  OCR0B=62 ;
  OCR0A=125-1 ; // sample period predivider 


  }




//----------------------------------------------------------------------------------------  

int main(){

  DDRD =0b11111111 ; // enable outputs , bit 0,1=input
  PORTD=0b11111111 ; // pullups on on input pins
  
  DDRB =0b11111111 ; // enable outputs 
  PORTB=0b11111111 ; // pullups on on input pins
  
  DDRC =0b11111110 ; // enable outputs , bit 0=input
  PORTC=0b11111110 ; // pullups off on ADC pins

//-------------------------------------------

  // ADMUX =0b01000000 ; // Vref=AVcc
  ADMUX =0b11000000 ; // Vref=1.1V

  ADCSRA=0b10100111 ; // 20MHz/128=156.25kHz    /13=12.019kHz = max sample frequency
  ADCSRB=0b00000110 ; // timer/counter1 overflow is trigger

 
// 20.00MHz/19200/16=65
// use 19200 bit/sec UBRRL=64
  UCSR0B |= ( 1 << TXEN0 );					// UART TX einschalten
  // UCSR0C |= ( 1 << URSEL )|( 3<<UCSZ0 );	    // Asynchron 8N1
  UBRR0H  = 0;                               // Highbyte ist 0
  UBRR0L  = 65-1 ;  // 19200 Bit/sec
  
  UBRR0L  = 22-1 ; //57600 Bd                        
  uartFIFOinit() ;
  timer0Init() ;
  timer1Init() ;
  timer2Init() ;

 

  CICfifoINIT() ;
  FRQdemodInit() ;

  TDFinit() ;

  uartCrlf() ;
  uartPutsPgm(PSTR(HEADLINE)) ;
  uartCrlf() ;

  sei() ;
  while (1){ 
    if ( IntFifoInPtr!=IntFifoOutPtr ) { 
      doIQsample() ; 
	  doClipCheck() ;
	  }
    if ( ClkMail500 ) { 
      ClkMail500=0 ; 
      TDFdemod() ;
      } ;
    }
  return(0) ;
  }
